//Wire library for I2C communication for debugging purposes
#include <Wire.h>

//Variable read in from python
int oscillation_frequency=50; //50mHz is default

//Variables for timing
unsigned long currentMillis, setup_time, timer1 = 0, timer2 = 0;
boolean flip_valve = true;

//Variables for reading flowrate and for smoothing function 
const int numReadings = 5;
float readings[numReadings];      // the readings from the analog input
int readIndex = 0;              // the index of the current reading
float total = 0, average = 0, flow_rate;  

/*
  How to hook it up correctly:
  |   Arduino PIN    |   Motor Driver PIN   |         Purpose      - Color Wire  |
  |        A0        |       N/A            |        Flow Meter Signal           |
  |        3         |         ENB          |       Speed Control  - black       |
  |        2        |         IN3          |          Close       - white       |
  |        4         |         IN4          |           Open       - gray        |
  |       11         |         N/A          |           Pressure Relief          |


  //debugging (optional): hook another arduino up to I2C lines to listen to I2C ouput
  //must load wire libraries example called "Slave Receiver" found in second example on:
  //https://www.arduino.cc/en/Tutorial/MasterWriter to second arduino to read data output
  |    This Arduino PIN      |   2nd Arduino Pins   |         Purpose           |
  |        A4 or SDA        |         A4 or SDA          |       SDA                  |
  |        A5 or SCL        |         A5 or SCL          |       SCL                  |
  
*/
boolean output_debugging_messages = false;

int Oscillation_Motor_Speed_PIN = 3;
int Oscillation_Motor_Open_PIN = 4;
int Oscillation_Motor_Close_PIN = 2;
int flowthrough_PIN = 0;  //A0
int pressure_PIN = 11;


void setup() {
  if(output_debugging_messages)
    Wire.begin(); // join i2c bus for debugging purposes
  
  //begin serial communication with python
  Serial.begin(9600);

  //set the speed pins to output a pulse width modulated signal to slow the motors speed
  pinMode(Oscillation_Motor_Speed_PIN, OUTPUT);
  analogWrite(Oscillation_Motor_Speed_PIN, 255); //0-255 180 is the slowest

  //set the control pins as outputs
  pinMode(Oscillation_Motor_Open_PIN, OUTPUT);
  pinMode(Oscillation_Motor_Close_PIN, OUTPUT);
  pinMode(pressure_PIN,INPUT);
  pinMode(LED_BUILTIN, OUTPUT); 

  //open the motors all the way so that flowmeters can be adjusted
  digitalWrite(Oscillation_Motor_Open_PIN, HIGH);
  delay(8000);
  digitalWrite(Oscillation_Motor_Open_PIN, LOW);

  //tell python Arduiono 2's while loop is beginning
  boolean initial_transmission_not_sent = true;
  while(initial_transmission_not_sent){
    delay(1000);
    Serial.println("Arduino 2 Start");
    String initial_response = Serial.readString();
    Serial.println(initial_response);
    if(initial_response == "ACK"){
        initial_transmission_not_sent = false;
     }
  }

  //initialize the smoothing array used for averaging the flow through sensor values
  for (int thisReading = 0; thisReading < numReadings; thisReading++) {
    readings[thisReading] = 0;
  }

  delay(5000);

  if(output_debugging_messages){
    //Tell debugging arduino we are done setting up
    Wire.beginTransmission(8); // transmit to device #8
    Wire.write("Done setting up ");        
    Wire.endTransmission();    // stop transmitting
  }

  //number of milliseconds elapsed since start of program to end of setup()
  setup_time = millis();
  timer1 = setup_time;

}

void loop() {
  //read the flowmeter 
  float flow_rate = readLPM();
  
  //read in serial from python if available
  send_read_serial_COM(flow_rate);

  //get current time elapsed since start of void loop()
  currentMillis = millis()-setup_time;

  //Controls
  if(digitalRead(pressure_PIN) == HIGH){
    open_valve();
    digitalWrite(LED_BUILTIN, HIGH); 
  }
  else{
    close_valve();
    digitalWrite(LED_BUILTIN, LOW); 
  }
  
}


void close_valve(){
  digitalWrite(Oscillation_Motor_Open_PIN, LOW);
  digitalWrite(Oscillation_Motor_Close_PIN, HIGH);
}

void open_valve(){
  digitalWrite(Oscillation_Motor_Close_PIN, LOW);
  digitalWrite(Oscillation_Motor_Open_PIN, HIGH);
}


void send_read_serial_COM(float flow_rate){

  //enter while loop every 5 seconds
  if(currentMillis - timer2 >= 5000UL){
    if(output_debugging_messages){
      //Tell debugging arduino we are done setting up
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Sending: ");
      Wire.write(byte(flow_rate));        
      Wire.endTransmission();    // stop transmitting
    }
    boolean keep_sending = true;
    while(keep_sending){
      Serial.println(flow_rate);
      delay(400);
      if (Serial.available()) { //there are new values to read
        String myString = Serial.readString();
        oscillation_frequency = getValue(myString, ',', 0).toInt();
        String acknowledge = getValue(myString, ',', 1);
        if(acknowledge == "ACK"){
          keep_sending = false; //exit while loop
        }
        else{
          delay(100); //keep sending
        }
      }
      
    }
    if(output_debugging_messages){
      //Tell debugging arduino we are done setting up
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Got: ");
      Wire.write(byte(oscillation_frequency));        
      Wire.endTransmission();    // stop transmitting
    }
    timer2 = currentMillis;   
  }
} 


//This function reads the analog input voltage of the flowmeter.
//It then averages 5 read values and
//returns the L/min value as a float.
float readLPM(){
  // subtract the last reading:
  total = total - readings[readIndex];
  // read from the sensor:
  readings[readIndex] = analogRead(flowthrough_PIN);
  // add the reading to the total:
  total = total + readings[readIndex];
  // advance to the next position in the array:
  readIndex++;
  // if we're at the end of the array...
  if (readIndex >= numReadings) {
    // ...wrap around to the beginning:
    readIndex = 0;
  }
  // calculate the average:
  average = total / numReadings;
  //convert ADC output to voltage
  average = average*(5.0/1024.0);
  //convert voltage to L/min using linear fit of datasheet values
  float average_flow_rate = (average-0.9833)/0.405;
  //makes sure positive (else = 0 L/min)
  if(average_flow_rate < 0){
    average_flow_rate = 0.0;
  }
  return average_flow_rate;
}

//Function to parse a string and return an int 
String getValue(String data, char separator, int index)
{
    int found = 0;
    int strIndex[] = { 0, -1 };
    int maxIndex = data.length() - 1;
    for (int i = 0; i <= maxIndex && found <= index; i++) {
        if (data.charAt(i) == separator || i == maxIndex) {
            found++;
            strIndex[0] = strIndex[1] + 1;
            strIndex[1] = (i == maxIndex) ? i+1 : i;
        }
    }
    return found > index ? data.substring(strIndex[0], strIndex[1]) : "";
}


